/// <summary>
/// Nutty Software Open WebGL Framework
/// 
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// This scene demonstrates how to take an arbitrary image and apply an old film effect.
/// </summary>


/// <summary>
/// Constructor.
/// </summary>
function OldFilmScene ()
{
	/// <summary>
	/// Setup inherited members.
	/// </summary>
	BaseScene.call(this);
	
	
	/// <summary>
	/// Old film shader program.
	/// </summary>
	this.mOldFilmShader = null;
	
	
	/// <summary>
	/// Texture image to operate on. In a real-time game, you would render your
	/// scene to this texture using an FBO (framebuffer object).
	/// </summary>
	this.mTexture = null;
}


/// <summary>
/// Prototypal Inheritance.
/// </summary>
OldFilmScene.prototype = new BaseScene();
OldFilmScene.prototype.constructor = OldFilmScene;


/// <summary>
/// Implementation.
/// </summary>
OldFilmScene.prototype.Start = function ()
{
	// Prepare resources to download
	this.mResource.Add(new ResourceItem("oldfilm.vs", null, "./shaders/oldfilm.vs"));
	this.mResource.Add(new ResourceItem("oldfilm.fs", null, "./shaders/oldfilm.fs"));
	this.mResource.Add(new ResourceItem("image", null, "./images/TorontoGooderham.jpg"));
	
	// Create scene content
	var rectMesh = new Rectangle(1.0, 1.0);
	var recVbo = new GLVertexBufferObject();
	recVbo.Create(rectMesh);
	
	// Create rectangular entity for post-processing the image.
	var rectEntity = new Entity();
	rectEntity.ObjectEntity = recVbo;
	rectEntity.ObjectMatrix.Scale(1.0, -1.0, 1.0); // Flip y-axis texture coordinates.
	this.mEntity.push(rectEntity);
	
	// Start downloading resources
	BaseScene.prototype.Start.call(this);
	
	// Setup user interface
	var sepiaSlider = $("#SepiaSlider");
	sepiaSlider.slider(
	{
		animate: true,
		orientation: "horizontal",
		range: "min",
		step: 0.01,
		max: 1.0,
		value: 0.6
	});
	sepiaSlider.on("slide", {owner : this}, this.SepiaValueChanged);
	sepiaSlider.on("slidechange", {owner : this}, this.SepiaValueChanged);
	
	
	var effectSlider = $("#NoiseSlider");
	effectSlider.slider(
	{
		animate: true,
		orientation: "horizontal",
		range: "min",
		step: 0.01,
		max: 1.0,
		value: 0.2
	});
	effectSlider.on("slide", {owner : this}, this.NoiseValueChanged);
	effectSlider.on("slidechange", {owner : this}, this.NoiseValueChanged);
	
	
	var effectSlider = $("#ScratchSlider");
	effectSlider.slider(
	{
		animate: true,
		orientation: "horizontal",
		range: "min",
		step: 0.01,
		max: 1.0,
		value: 0.3
	});
	effectSlider.on("slide", {owner : this}, this.ScratchValueChanged);
	effectSlider.on("slidechange", {owner : this}, this.ScratchValueChanged);
	
	
	var effectSlider = $("#VignettingSlider");
	effectSlider.slider(
	{
		animate: true,
		orientation: "horizontal",
		range: "min",
		step: 0.01,
		max: 1.0,
		value: 0.3
	});
	effectSlider.on("slide", {owner : this}, this.VignettingValueChanged);
	effectSlider.on("slidechange", {owner : this}, this.VignettingValueChanged);
}


OldFilmScene.prototype.SepiaValueChanged = function (event, ui)
{
	if ( event.data.owner.mOldFilmShader )
		event.data.owner.mOldFilmShader.SepiaValue = ui.value;
}


OldFilmScene.prototype.NoiseValueChanged = function (event, ui)
{
	if ( event.data.owner.mOldFilmShader )
		event.data.owner.mOldFilmShader.NoiseValue = ui.value;
}


OldFilmScene.prototype.ScratchValueChanged = function (event, ui)
{
	if ( event.data.owner.mOldFilmShader )
		event.data.owner.mOldFilmShader.ScratchValue = ui.value;
}


OldFilmScene.prototype.VignettingValueChanged = function (event, ui)
{
	if ( event.data.owner.mOldFilmShader )
		event.data.owner.mOldFilmShader.VignettingValue = ui.value;
}


/// <summary>
/// Implementation.
/// </summary>
OldFilmScene.prototype.Update = function ()
{
	BaseScene.prototype.Update.call(this);
	
	// Draw only when all resources have been downloaded
	if ( this.mOldFilmShader )
	{
		this.mOldFilmShader.Enable();
		this.mOldFilmShader.RandomValue = Math.random();
		
		for (var i = 0; i < this.mEntity.length; ++i)
			this.mOldFilmShader.Draw(this.mEntity[i]);
			
		this.mOldFilmShader.TimeLapse += 1.0;
		
		this.mOldFilmShader.Disable();
	}
}


/// <summary>
/// Implementation.
/// </summary>
OldFilmScene.prototype.End = function ()
{
	BaseScene.prototype.End.call(this);

	// Cleanup
	if ( this.mOldFilmShader )
	{
		this.mOldFilmShader.Release();
		this.mOldFilmShader = null;
	}
	
	if ( this.mTexture )
	{
		this.mTexture.Release();
		this.mTexture = null;
	}
}



/// <summary>
/// Implementation.
/// </summary>
OldFilmScene.prototype.OnLoadComplete = function ()
{
	// Get images
	var image = this.mResource.Find("image");

	// Process shaders
	var oldFilmVs = this.mResource.Find("oldfilm.vs");
	var oldFilmFs = this.mResource.Find("oldfilm.fs");

	
	if ( (oldFilmVs == null) || (oldFilmVs.Item == null) ||
		 (oldFilmFs == null) || (oldFilmFs.Item == null) ||
		 (image == null) )
	{
		// Resources missing?
	}
	else
	{
		// Load texture image
		this.mTexture = new GLTexture2D();
		this.mTexture.Create(image.Item.width, image.Item.height, Texture.Format.Rgb, SamplerState.LinearClamp, image.Item);
		this.mEntity[0].ObjectMaterial.Texture = new Array();
		this.mEntity[0].ObjectMaterial.Texture[0] = this.mTexture;
	
		// Load vertex shader
		var shaderVs = new GLShader();
		if ( !shaderVs.Create(shaderVs.ShaderType.Vertex, oldFilmVs.Item) )
		{
			// Report error
			var log = shaderVs.GetLog();
			alert("Error compiling " + oldFilmVs.Name + ".\n\n" + log);
		}
		else
			this.mShader.push(shaderVs);
		
		// Load fragment shader
		var shaderFs = new GLShader();
		if ( !shaderFs.Create(shaderFs.ShaderType.Fragment, oldFilmFs.Item) )
		{
			// Report error
			var log = shaderFs.GetLog();
			alert("Error compiling " + oldFilmFs.Name + ".\n\n" + log);
		}
		else
			this.mShader.push(shaderFs);
			
		// Update view matrix
		this.mViewMatrix.Translate(0.0, 0.0, 1.0);
		
		// Create shader program
		this.mOldFilmShader = new OldFilmShader();
		this.mOldFilmShader.Projection = this.mOrthographicMatrix;
		this.mOldFilmShader.View = this.mViewMatrix.Inverse();
		this.mOldFilmShader.Create();
		this.mOldFilmShader.AddShader(shaderVs);
		this.mOldFilmShader.AddShader(shaderFs);
		this.mOldFilmShader.Link();
		this.mOldFilmShader.Init();
		this.mOldFilmShader.SepiaValue = 0.6;
		this.mOldFilmShader.NoiseValue = 0.2;
		
		var canvas = document.getElementById("Canvas");
		this.mOldFilmShader.SetSize(canvas.width, canvas.height);
	}
}